<?php

namespace Tests\Feature;

use App\Models\Attachment;
use App\Models\Company;
use App\Models\Occurrence;
use App\Models\OccurrenceType;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Tests\TestCase;

class AttachmentDownloadTest extends TestCase
{
    use RefreshDatabase;

    public function test_superadmin_can_view_company_panel_and_download_attachment()
    {
        Storage::fake('public');

        $admin = User::factory()->create(['role' => 'superadmin']);
        $company = Company::create(['name' => 'ACME']);
        $type = OccurrenceType::create(['name' => 'Call']);

        $this->actingAs($admin)
            ->postJson('/occurrences', [
                'company_id' => $company->id,
                'occurrence_type_id' => $type->id,
                'details' => 'Download test',
                'attachments' => [UploadedFile::fake()->create('doc.txt', 10)],
            ])
            ->assertStatus(201)
            ->assertJsonFragment(['details' => 'Download test']);

        $occurrence = Occurrence::first();
        $attachment = Attachment::first();

        // Company panel should return the view with occurrences
        $this->actingAs($admin)
            ->get('/company/'.$company->id.'/panel')
            ->assertStatus(200)
            ->assertSee('Download');

        // Download should be allowed for superadmin
        $this->actingAs($admin)
            ->get('/attachments/'.$attachment->id.'/download')
            ->assertStatus(200);
    }

    public function test_employee_cannot_download_other_users_attachment()
    {
        Storage::fake('public');

        $admin = User::factory()->create(['role' => 'superadmin']);
        $employee = User::factory()->create(['role' => 'employee']);
        $company = Company::create(['name' => 'ACME']);
        $type = OccurrenceType::create(['name' => 'Visit']);

        // Admin creates an occurrence with attachment
        $this->actingAs($admin)
            ->postJson('/occurrences', [
                'company_id' => $company->id,
                'occurrence_type_id' => $type->id,
                'details' => 'Admin occurrence',
                'attachments' => [UploadedFile::fake()->create('secret.txt', 10)],
            ])
            ->assertStatus(201);

        $attachment = Attachment::first();

        // Employee should not be able to download attachment they don't own
        $this->actingAs($employee)
            ->get('/attachments/'.$attachment->id.'/download')
            ->assertStatus(403);
    }
}
