<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Company;
use App\Models\OccurrenceType;
use Illuminate\Support\Facades\Auth;

class EmployeeController extends Controller
{
    public function companies()
    {
        $companies = Company::paginate(15);
        return view('employee.companies', compact('companies'));
    }

    public function dashboard()
    {
        $user = Auth::user();
        $userId = $user->id;
        // Mirror client dashboard data but scoped to occurrences created by this user
        $days = 30;
        $labels = [];
        $dateStart = now()->subDays($days - 1)->startOfDay();
        for ($i = 0; $i < $days; $i++) {
            $labels[] = $dateStart->copy()->addDays($i)->format('d/m');
        }

        $chartCalls = ['labels' => $labels, 'data' => array_fill(0, $days, 0)];
        $chartEmails = ['labels' => $labels, 'data' => array_fill(0, $days, 0)];
        $chartDocs = ['labels' => $labels, 'data' => array_fill(0, $days, 0)];

        $callType = \App\Models\OccurrenceType::whereRaw('lower(name) = ?', ['chamadas'])->orWhere('name','Chamadas')->first();
        $emailType = \App\Models\OccurrenceType::whereRaw('lower(name) = ?', ['emails'])->orWhere('name','Emails')->first();
        $receptionType = \App\Models\OccurrenceType::whereRaw('lower(name) = ?', ['recepção'])->orWhere('name','Recepção')->first();

        $callTypeId = $callType->id ?? null;
        $emailTypeId = $emailType->id ?? null;
        $receptionTypeId = $receptionType->id ?? null;

        foreach (['calls' => $callTypeId, 'emails' => $emailTypeId] as $key => $typeId) {
            if ($typeId) {
                $rows = \App\Models\Occurrence::where('user_id', $userId)
                    ->where('occurrence_type_id', $typeId)
                    ->where('created_at', '>=', $dateStart)
                    ->selectRaw("DATE(created_at) as day, count(*) as total")
                    ->groupBy('day')
                    ->get();

                foreach ($rows as $r) {
                    $d = \Carbon\Carbon::parse($r->day)->format('d/m');
                    $index = array_search($d, $labels);
                    if ($index !== false) {
                        if ($key === 'calls') $chartCalls['data'][$index] = (int) $r->total;
                        if ($key === 'emails') $chartEmails['data'][$index] = (int) $r->total;
                    }
                }
            }
        }

        $rows = \App\Models\Attachment::whereHas('occurrence', fn($q) => $q->where('user_id', $userId))
            ->where('created_at', '>=', $dateStart)
            ->selectRaw("DATE(created_at) as day, count(*) as total")
            ->groupBy('day')
            ->get();

        foreach ($rows as $r) {
            $d = \Carbon\Carbon::parse($r->day)->format('d/m');
            $index = array_search($d, $labels);
            if ($index !== false) $chartDocs['data'][$index] = (int) $r->total;
        }

        // counts - occurrences created by this user grouped by type
        $counts = \App\Models\Occurrence::where('user_id', $userId)->selectRaw('occurrence_type_id, count(*) as total')->groupBy('occurrence_type_id')->pluck('total','occurrence_type_id')->toArray();

        $attachmentsCount = \App\Models\Attachment::whereHas('occurrence', fn($q) => $q->where('user_id', $userId))->count();

        // Compute deltas similar to client (compare yesterday vs day before)
        $yesterday = now()->subDays(1)->startOfDay();
        $dayBefore = now()->subDays(2)->startOfDay();

        $deltaForType = function($typeId) use ($userId, $yesterday, $dayBefore) {
            if (! $typeId) return ['value' => 0, 'percent' => 0, 'trend' => 'same'];

            $last = \App\Models\Occurrence::where('user_id', $userId)
                ->where('occurrence_type_id', $typeId)
                ->where('created_at', '>=', $yesterday)
                ->where('created_at', '<', $yesterday->copy()->addDay())
                ->count();

            $prev = \App\Models\Occurrence::where('user_id', $userId)
                ->where('occurrence_type_id', $typeId)
                ->where('created_at', '>=', $dayBefore)
                ->where('created_at', '<', $dayBefore->copy()->addDay())
                ->count();

            $percent = 0;
            $trend = 'same';
            if ($prev == 0 && $last > 0) {
                $percent = 100;
                $trend = 'up';
            } elseif ($prev == 0 && $last == 0) {
                $percent = 0;
                $trend = 'same';
            } else {
                $diff = $last - $prev;
                $percent = $prev ? round(($diff / max(1, $prev)) * 100) : 0;
                $trend = $diff > 0 ? 'up' : ($diff < 0 ? 'down' : 'same');
            }

            return ['value' => $last, 'percent' => $percent, 'trend' => $trend];
        };

        $deltaCalls = $deltaForType($callTypeId);
        $deltaEmails = $deltaForType($emailTypeId);
        $deltaReception = $deltaForType($receptionTypeId);

        // attachments delta
        $lastAtt = \App\Models\Attachment::whereHas('occurrence', fn($q) => $q->where('user_id', $userId))
            ->where('created_at', '>=', $yesterday)
            ->where('created_at', '<', $yesterday->copy()->addDay())
            ->count();
        $prevAtt = \App\Models\Attachment::whereHas('occurrence', fn($q) => $q->where('user_id', $userId))
            ->where('created_at', '>=', $dayBefore)
            ->where('created_at', '<', $dayBefore->copy()->addDay())
            ->count();

        $attDiff = $lastAtt - $prevAtt;
        if ($prevAtt == 0 && $lastAtt > 0) {
            $attPercent = 100; $attTrend = 'up';
        } elseif ($prevAtt == 0 && $lastAtt == 0) {
            $attPercent = 0; $attTrend = 'same';
        } else {
            $attPercent = $prevAtt ? round(($attDiff / max(1,$prevAtt)) * 100) : 0;
            $attTrend = $attDiff > 0 ? 'up' : ($attDiff < 0 ? 'down' : 'same');
        }

        $deltaAttachments = ['value' => $lastAtt, 'percent' => $attPercent, 'trend' => $attTrend];

        return view('employee.dashboard', compact('chartCalls','chartEmails','chartDocs','counts','attachmentsCount','callTypeId','emailTypeId','receptionTypeId','deltaCalls','deltaEmails','deltaReception','deltaAttachments'));
    }

    public function createOccurrenceForm(Company $company)
    {
        $types = OccurrenceType::all();
        return view('employee.create_occurrence', compact('company', 'types'));
    }

    protected function typeList($typeName)
    {
        $user = Auth::user();
        // Robust lookup: normalize names and allow fallback creation (matches OccurrenceController behaviour)
        $normalize = fn($s) => \Illuminate\Support\Str::ascii(strtolower($s));
        $target = $normalize($typeName);

        $occType = null;
        foreach (\App\Models\OccurrenceType::all() as $t) {
            $n = $normalize($t->name);
            $nTrim = rtrim($n, 's');
            $tTrim = rtrim($target, 's');
            if ($n === $target || strpos($n, $target) !== false || $nTrim === $tTrim) {
                $occType = $t;
                break;
            }
        }

        if (! $occType) {
            $fallbacks = [
                'chamadas' => 'Chamadas',
                'emails' => 'Emails',
                'recepcao' => 'Recepção',
            ];

            if (isset($fallbacks[$target])) {
                $occType = \App\Models\OccurrenceType::firstOrCreate(['name' => $fallbacks[$target]], ['description' => 'Auto-created']);
            }
        }

        if (! $occType) abort(404);

        $occurrences = \App\Models\Occurrence::where('occurrence_type_id', $occType->id)->where('user_id', $user->id)->with(['company','attachments'])->orderBy('created_at','desc')->paginate(15);

        $companies = \App\Models\Company::all();

        return view('employee.occurrences', compact('occType','occurrences','companies'));
        }

        public function chamadas()
        {
            return $this->typeList('Chamadas');
        }

        public function emails()
        {
            return $this->typeList('Emails');
        }

        public function recepcao()
        {
            return $this->typeList('Recepção');
        }

        public function showOccurrence(\App\Models\Occurrence $occurrence)
        {
            // employees can view only occurrences they created
            $user = Auth::user();
            if ($occurrence->user_id !== $user->id && $user->role !== 'superadmin') abort(403);
            $occurrence->load(['user','attachments','type','company']);
            return view('employee.occurrence_show', compact('occurrence'));
        }
    }
