<?php

namespace App\Http\Controllers;

use App\Models\Company;
use Illuminate\Http\Request;

use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;

class CompanyController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->authorizeResource(Company::class, 'company');
    }

    public function index()
    {
        $user = Auth::user();
        if ($user && $user->role === 'superadmin') {
            return Company::all();
        }

        return Company::where('created_by', $user->id)->get();
    }

    public function store(Request $request)
    {
        $this->authorize('create', Company::class);

        $data = $request->validate([
            // company
            'name' => 'required|string|max:255',
            'identifier' => 'nullable|string|max:100',
            // client user
            'user_name' => 'required|string|max:255',
            'user_email' => 'required|email|max:255|unique:users,email',
            'user_password' => 'required|confirmed|min:6',
        ]);

        // create company and client user atomically
        return DB::transaction(function() use ($data) {
            $user = \App\Models\User::create([
                'name' => $data['user_name'],
                'email' => $data['user_email'],
                'password' => Hash::make($data['user_password']),
                'role' => 'client',
                // mark email as verified for now
                'email_verified_at' => now(),
            ]);

            $company = Company::create([
                'name' => $data['name'],
                'identifier' => $data['identifier'] ?? null,
                'created_by' => $user->id,
            ]);

            return redirect()->route('company.panel', $company->id);
        });
    }

    /**
     * Show the form to create a new company (and its admin user) - for employees/superadmin.
     */
    public function create()
    {
        $this->authorize('create', Company::class);
        return view('company.create');
    }

        /**
         * AJAX search used by Select2
         */
        public function search(Request $request)
        {
            $q = $request->get('q');
            $page = max(1, (int) $request->get('page', 1));
            $perPage = 20;

            $query = Company::query();
            if ($q) {
                $query->where('name', 'like', "%{$q}%");
            }

            $total = $query->count();
            $companies = $query->orderBy('name')->skip(($page - 1) * $perPage)->take($perPage)->get();

            $results = $companies->map(function($c){ return ['id' => $c->id, 'text' => $c->name]; });

            return response()->json([
                'results' => $results,
                'pagination' => ['more' => ($page * $perPage) < $total]
            ]);
        }
    /**
     * Show the company panel with occurrences.
     */
    public function panel(Company $company)
    {
        $this->authorize('view', $company);

    $user = \Illuminate\Support\Facades\Auth::user();
    $userId = $user->id;

        // Prepare same dashboard data as EmployeeController but scoped to this company + current user
        $days = 30;
        $labels = [];
        $dateStart = now()->subDays($days - 1)->startOfDay();
        for ($i = 0; $i < $days; $i++) {
            $labels[] = $dateStart->copy()->addDays($i)->format('d/m');
        }

        $chartCalls = ['labels' => $labels, 'data' => array_fill(0, $days, 0)];
        $chartEmails = ['labels' => $labels, 'data' => array_fill(0, $days, 0)];
        $chartDocs = ['labels' => $labels, 'data' => array_fill(0, $days, 0)];

        $callType = \App\Models\OccurrenceType::whereRaw('lower(name) = ?', ['chamadas'])->orWhere('name','Chamadas')->first();
        $emailType = \App\Models\OccurrenceType::whereRaw('lower(name) = ?', ['emails'])->orWhere('name','Emails')->first();
        $receptionType = \App\Models\OccurrenceType::whereRaw('lower(name) = ?', ['recepção'])->orWhere('name','Recepção')->first();

        $callTypeId = $callType->id ?? null;
        $emailTypeId = $emailType->id ?? null;
        $receptionTypeId = $receptionType->id ?? null;

        foreach (['calls' => $callTypeId, 'emails' => $emailTypeId] as $key => $typeId) {
            if ($typeId) {
                $rows = \App\Models\Occurrence::where('company_id', $company->id)
                    ->where('user_id', $userId)
                    ->where('occurrence_type_id', $typeId)
                    ->where('created_at', '>=', $dateStart)
                    ->selectRaw("DATE(created_at) as day, count(*) as total")
                    ->groupBy('day')
                    ->get();

                foreach ($rows as $r) {
                    $d = \Carbon\Carbon::parse($r->day)->format('d/m');
                    $index = array_search($d, $labels);
                    if ($index !== false) {
                        if ($key === 'calls') $chartCalls['data'][$index] = (int) $r->total;
                        if ($key === 'emails') $chartEmails['data'][$index] = (int) $r->total;
                    }
                }
            }
        }

        $rows = \App\Models\Attachment::whereHas('occurrence', fn($q) => $q->where('company_id', $company->id)->where('user_id', $userId))
            ->where('created_at', '>=', $dateStart)
            ->selectRaw("DATE(created_at) as day, count(*) as total")
            ->groupBy('day')
            ->get();

        foreach ($rows as $r) {
            $d = \Carbon\Carbon::parse($r->day)->format('d/m');
            $index = array_search($d, $labels);
            if ($index !== false) $chartDocs['data'][$index] = (int) $r->total;
        }

        // counts (occurrences for this company created by this user)
        $counts = \App\Models\Occurrence::where('company_id', $company->id)->where('user_id', $userId)->selectRaw('occurrence_type_id, count(*) as total')->groupBy('occurrence_type_id')->pluck('total','occurrence_type_id')->toArray();

        $attachmentsCount = \App\Models\Attachment::whereHas('occurrence', fn($q) => $q->where('company_id', $company->id)->where('user_id', $userId))->count();

        // deltas
        $yesterday = now()->subDays(1)->startOfDay();
        $dayBefore = now()->subDays(2)->startOfDay();

        $deltaForType = function($typeId) use ($company, $userId, $yesterday, $dayBefore) {
            if (! $typeId) return ['value' => 0, 'percent' => 0, 'trend' => 'same'];

            $last = \App\Models\Occurrence::where('company_id', $company->id)->where('user_id', $userId)
                ->where('occurrence_type_id', $typeId)
                ->where('created_at', '>=', $yesterday)
                ->where('created_at', '<', $yesterday->copy()->addDay())
                ->count();

            $prev = \App\Models\Occurrence::where('company_id', $company->id)->where('user_id', $userId)
                ->where('occurrence_type_id', $typeId)
                ->where('created_at', '>=', $dayBefore)
                ->where('created_at', '<', $dayBefore->copy()->addDay())
                ->count();

            $percent = 0; $trend = 'same';
            if ($prev == 0 && $last > 0) { $percent = 100; $trend = 'up'; }
            elseif ($prev == 0 && $last == 0) { $percent = 0; $trend = 'same'; }
            else { $diff = $last - $prev; $percent = $prev ? round(($diff / max(1, $prev)) * 100) : 0; $trend = $diff > 0 ? 'up' : ($diff < 0 ? 'down' : 'same'); }

            return ['value' => $last, 'percent' => $percent, 'trend' => $trend];
        };

        $deltaCalls = $deltaForType($callTypeId);
        $deltaEmails = $deltaForType($emailTypeId);
        $deltaReception = $deltaForType($receptionTypeId);

        $lastAtt = \App\Models\Attachment::whereHas('occurrence', fn($q) => $q->where('company_id', $company->id)->where('user_id', $userId))
            ->where('created_at', '>=', $yesterday)
            ->where('created_at', '<', $yesterday->copy()->addDay())
            ->count();
        $prevAtt = \App\Models\Attachment::whereHas('occurrence', fn($q) => $q->where('company_id', $company->id)->where('user_id', $userId))
            ->where('created_at', '>=', $dayBefore)
            ->where('created_at', '<', $dayBefore->copy()->addDay())
            ->count();

        $attDiff = $lastAtt - $prevAtt;
        if ($prevAtt == 0 && $lastAtt > 0) { $attPercent = 100; $attTrend = 'up'; }
        elseif ($prevAtt == 0 && $lastAtt == 0) { $attPercent = 0; $attTrend = 'same'; }
        else { $attPercent = $prevAtt ? round(($attDiff / max(1,$prevAtt)) * 100) : 0; $attTrend = $attDiff > 0 ? 'up' : ($attDiff < 0 ? 'down' : 'same'); }

        $deltaAttachments = ['value' => $lastAtt, 'percent' => $attPercent, 'trend' => $attTrend];

        // also include attachments for this company so the panel can show download links
        $attachments = \App\Models\Attachment::whereHas('occurrence', fn($q) => $q->where('company_id', $company->id))->get();

        return view('employee.dashboard', compact('chartCalls','chartEmails','chartDocs','counts','attachmentsCount','callTypeId','emailTypeId','receptionTypeId','deltaCalls','deltaEmails','deltaReception','deltaAttachments','company','attachments'));
    }

    /**
     * Company dashboard: show counters grouped by occurrence type
     */
    public function dashboard(Company $company)
    {
        $this->authorize('view', $company);

        // get all occurrence types
        $types = \App\Models\OccurrenceType::all();

        // counts per type for this company
        $counts = $company->occurrences()
            ->selectRaw('occurrence_type_id, count(*) as total')
            ->groupBy('occurrence_type_id')
            ->pluck('total', 'occurrence_type_id')
            ->toArray();

        return view('company.dashboard', compact('company', 'types', 'counts'));
    }

    /**
     * Show occurrences for a company filtered by occurrence type (id or name)
     */
    public function occurrencesByType(Company $company, $type)
    {
        $this->authorize('view', $company);

        // resolve type by id or name
        $occType = \App\Models\OccurrenceType::where('id', $type)
            ->orWhere('name', $type)
            ->first();

        if (! $occType) {
            abort(404);
        }

        $occurrences = $company->occurrences()
            ->where('occurrence_type_id', $occType->id)
            ->with(['type', 'user', 'attachments'])
            ->orderBy('created_at', 'desc')
            ->paginate(15);

        return view('company.occurrences_by_type', compact('company', 'occType', 'occurrences'));
    }

    public function indexPanel()
    {
    $user = \Illuminate\Support\Facades\Auth::user();
        // Superadmin and employees (system operators) should see all companies.
        if ($user && in_array($user->role, ['superadmin', 'employee'])) {
            $companies = Company::all();
        } else {
            // clients (company admins) see only their own company
            $companies = Company::where('created_by', $user->id)->get();
        }

        return view('company.index', compact('companies'));
    }
}