<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ClientController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
        $this->middleware('role:client');
    }

    protected function companyForUser()
    {
        $user = Auth::user();
        return \App\Models\Company::where('created_by', $user->id)->first();
    }

    public function dashboard()
    {
        $company = $this->companyForUser();
        if (! $company) abort(404);
        $data = $this->dashboardDataForRender($company);

        extract($data);

        return view('client.dashboard', $data);
    }

    /**
     * Prepare dashboard data array for a given company. Used for rendering outside HTTP context.
     */
    public function dashboardDataForRender(\App\Models\Company $company): array
    {
        $types = \App\Models\OccurrenceType::all();
        $counts = $company->occurrences()->selectRaw('occurrence_type_id, count(*) as total')->groupBy('occurrence_type_id')->pluck('total','occurrence_type_id')->toArray();

        // Resolve common type ids by name (case-insensitive)
        $callType = \App\Models\OccurrenceType::whereRaw('lower(name) = ?', ['chamadas'])->orWhere('name','Chamadas')->first();
        $emailType = \App\Models\OccurrenceType::whereRaw('lower(name) = ?', ['emails'])->orWhere('name','Emails')->first();
        $receptionType = \App\Models\OccurrenceType::whereRaw('lower(name) = ?', ['recepção'])->orWhere('name','Recepção')->first();

        $callTypeId = $callType->id ?? null;
        $emailTypeId = $emailType->id ?? null;
        $receptionTypeId = $receptionType->id ?? null;

        // attachments count
        $attachmentsCount = \App\Models\Attachment::whereHas('occurrence', fn($q) => $q->where('company_id', $company->id))->count();

        // Prepare chart data for the last 30 days (calls, emails, docs)
        $days = 30;
        $labels = [];
        $dateStart = now()->subDays($days - 1)->startOfDay();
        for ($i = 0; $i < $days; $i++) {
            $labels[] = $dateStart->copy()->addDays($i)->format('d/m');
        }

        $chartCalls = ['labels' => $labels, 'data' => array_fill(0, $days, 0)];
        $chartEmails = ['labels' => $labels, 'data' => array_fill(0, $days, 0)];
        $chartDocs = ['labels' => $labels, 'data' => array_fill(0, $days, 0)];

        // Occurrences per day for calls and emails
        foreach (['calls' => $callTypeId, 'emails' => $emailTypeId] as $key => $typeId) {
            if ($typeId) {
                $rows = \App\Models\Occurrence::where('company_id', $company->id)
                    ->where('occurrence_type_id', $typeId)
                    ->where('created_at', '>=', $dateStart)
                    ->selectRaw("DATE(created_at) as day, count(*) as total")
                    ->groupBy('day')
                    ->get();

                foreach ($rows as $r) {
                    $d = \Carbon\Carbon::parse($r->day)->format('d/m');
                    $index = array_search($d, $labels);
                    if ($index !== false) {
                        if ($key === 'calls') $chartCalls['data'][$index] = (int) $r->total;
                        if ($key === 'emails') $chartEmails['data'][$index] = (int) $r->total;
                    }
                }
            }
        }

        // Attachments per day (documents)
        $rows = \App\Models\Attachment::whereHas('occurrence', fn($q) => $q->where('company_id', $company->id))
            ->where('created_at', '>=', $dateStart)
            ->selectRaw("DATE(created_at) as day, count(*) as total")
            ->groupBy('day')
            ->get();

        foreach ($rows as $r) {
            $d = \Carbon\Carbon::parse($r->day)->format('d/m');
            $index = array_search($d, $labels);
            if ($index !== false) {
                $chartDocs['data'][$index] = (int) $r->total;
            }
        }

        // Compute simple deltas: compare last day vs previous day for each metric
        $yesterday = now()->subDays(1)->startOfDay();
        $dayBefore = now()->subDays(2)->startOfDay();

        $deltaForType = function($typeId) use ($company, $yesterday, $dayBefore) {
            if (! $typeId) return ['value' => 0, 'percent' => 0, 'trend' => 'same'];

            $last = \App\Models\Occurrence::where('company_id', $company->id)
                ->where('occurrence_type_id', $typeId)
                ->where('created_at', '>=', $yesterday)
                ->where('created_at', '<', $yesterday->copy()->addDay())
                ->count();

            $prev = \App\Models\Occurrence::where('company_id', $company->id)
                ->where('occurrence_type_id', $typeId)
                ->where('created_at', '>=', $dayBefore)
                ->where('created_at', '<', $dayBefore->copy()->addDay())
                ->count();

            $percent = 0;
            $trend = 'same';
            if ($prev == 0 && $last > 0) {
                $percent = 100;
                $trend = 'up';
            } elseif ($prev == 0 && $last == 0) {
                $percent = 0;
                $trend = 'same';
            } else {
                $diff = $last - $prev;
                $percent = $prev ? round(($diff / max(1, $prev)) * 100) : 0;
                $trend = $diff > 0 ? 'up' : ($diff < 0 ? 'down' : 'same');
            }

            return ['value' => $last, 'percent' => $percent, 'trend' => $trend];
        };

        $deltaCalls = $deltaForType($callTypeId);
        $deltaEmails = $deltaForType($emailTypeId);
        $deltaReception = $deltaForType($receptionTypeId);

        // attachments delta
        $lastAtt = \App\Models\Attachment::whereHas('occurrence', fn($q) => $q->where('company_id', $company->id))
            ->where('created_at', '>=', $yesterday)
            ->where('created_at', '<', $yesterday->copy()->addDay())
            ->count();
        $prevAtt = \App\Models\Attachment::whereHas('occurrence', fn($q) => $q->where('company_id', $company->id))
            ->where('created_at', '>=', $dayBefore)
            ->where('created_at', '<', $dayBefore->copy()->addDay())
            ->count();

        $attDiff = $lastAtt - $prevAtt;
        if ($prevAtt == 0 && $lastAtt > 0) {
            $attPercent = 100; $attTrend = 'up';
        } elseif ($prevAtt == 0 && $lastAtt == 0) {
            $attPercent = 0; $attTrend = 'same';
        } else {
            $attPercent = $prevAtt ? round(($attDiff / max(1,$prevAtt)) * 100) : 0;
            $attTrend = $attDiff > 0 ? 'up' : ($attDiff < 0 ? 'down' : 'same');
        }

        $deltaAttachments = ['value' => $lastAtt, 'percent' => $attPercent, 'trend' => $attTrend];

        return compact('company','types','counts','callTypeId','emailTypeId','receptionTypeId','attachmentsCount','chartCalls','chartEmails','chartDocs');
    }

    public function chamadas()
    {
        return $this->typeList('Chamadas');
    }

    public function emails()
    {
        return $this->typeList('Emails');
    }

    public function recepcao()
    {
        return $this->typeList('Recepção');
    }

    protected function typeList($typeName)
    {
        $company = $this->companyForUser();
        if (! $company) abort(404);

        // Robust lookup: normalize names and allow fallback creation (matches OccurrenceController behaviour)
        $normalize = fn($s) => \Illuminate\Support\Str::ascii(strtolower($s));
        $target = $normalize($typeName);

        $occType = null;
        foreach (\App\Models\OccurrenceType::all() as $t) {
            $n = $normalize($t->name);
            $nTrim = rtrim($n, 's');
            $tTrim = rtrim($target, 's');
            if ($n === $target || strpos($n, $target) !== false || $nTrim === $tTrim) {
                $occType = $t;
                break;
            }
        }

        if (! $occType) {
            $fallbacks = [
                'chamadas' => 'Chamadas',
                'emails' => 'Emails',
                'recepcao' => 'Recepção',
            ];

            if (isset($fallbacks[$target])) {
                $occType = \App\Models\OccurrenceType::firstOrCreate(['name' => $fallbacks[$target]], ['description' => 'Auto-created']);
            }
        }

        if (! $occType) abort(404);

        $occurrences = $company->occurrences()->where('occurrence_type_id', $occType->id)->with(['user','attachments'])->orderBy('created_at','desc')->paginate(15);

        return view('client.occurrences', compact('company','occType','occurrences'));
    }

    public function attachments()
    {
        $company = $this->companyForUser();
        if (! $company) abort(404);

        $attachments = \App\Models\Attachment::whereHas('occurrence', fn($q) => $q->where('company_id', $company->id))->with('occurrence')->orderBy('created_at','desc')->paginate(20);

        return view('client.attachments', compact('company','attachments'));
    }

    public function showOccurrence(\App\Models\Occurrence $occurrence)
    {
        $company = $this->companyForUser();
        if (! $company) abort(404);

        if ($occurrence->company_id !== $company->id) abort(403);

        $occurrence->load(['user','attachments','type']);

        // Note: do NOT change status on view. Status changes are now explicit actions.

        return view('client.occurrence_show', compact('company','occurrence'));
    }

    /**
     * POST handler: open occurrence (used when client clicks the "Ver" button from lists).
     * Sets status to 'open' if currently closed then redirects to the show page.
     */
    public function openOccurrence(\App\Models\Occurrence $occurrence)
    {
        $company = $this->companyForUser();
        if (! $company) abort(404);

        if ($occurrence->company_id !== $company->id) abort(403);

        if ($occurrence->status === 'closed') {
            $occurrence->status = 'open';
            $occurrence->save();
        }

        return redirect()->route('client.occurrence.show', $occurrence->id);
    }

    /**
     * POST handler: toggle occurrence status between 'open' and 'closed'.
     * Expects CSRF and client ownership. Redirects back to the occurrence show page.
     */
    public function toggleStatus(\Illuminate\Http\Request $request, \App\Models\Occurrence $occurrence)
    {
        $company = $this->companyForUser();
        if (! $company) abort(404);

        if ($occurrence->company_id !== $company->id) abort(403);

        $occurrence->status = ($occurrence->status === 'open') ? 'closed' : 'open';
        $occurrence->save();

        return redirect()->back()->with('message', 'Status atualizado.');
    }
}
