<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Company;
use App\Models\Occurrence;
use App\Models\OccurrenceType;
use App\Models\User;
use App\Models\Attachment;

class AdminController extends Controller
{
    public function dashboard()
    {
        $days = 30;
        $labels = [];
        $dateStart = now()->subDays($days - 1)->startOfDay();
        for ($i = 0; $i < $days; $i++) {
            $labels[] = $dateStart->copy()->addDays($i)->format('d/m');
        }

        // Counts
        $companiesCount = Company::count();
        $activeCompanies = Company::whereHas('occurrences', function($q) use ($dateStart) {
            $q->where('created_at', '>=', $dateStart);
        })->count();
        $employeesCount = User::where('role', 'employee')->count();
        $occurrencesCount = Occurrence::count();

        // Charts
        $chartCompanies = ['labels' => $labels, 'data' => array_fill(0, $days, 0)];
        $chartCalls = ['labels' => $labels, 'data' => array_fill(0, $days, 0)];
        $chartEmails = ['labels' => $labels, 'data' => array_fill(0, $days, 0)];

        // Companies created per day
        $rows = Company::where('created_at', '>=', $dateStart)
            ->selectRaw("DATE(created_at) as day, count(*) as total")
            ->groupBy('day')
            ->get();
        foreach ($rows as $r) {
            $d = \Carbon\Carbon::parse($r->day)->format('d/m');
            $index = array_search($d, $labels);
            if ($index !== false) $chartCompanies['data'][$index] = (int) $r->total;
        }

        // Flexible lookup: match common substrings (supports 'Chamadas', 'Call', 'Calls', 'Emails', etc.)
        $callType = OccurrenceType::whereRaw('lower(name) like ?', ['%chamad%'])
            ->orWhereRaw('lower(name) like ?', ['%call%'])->first();
        $emailType = OccurrenceType::whereRaw('lower(name) like ?', ['%email%'])->orWhereRaw('lower(name) like ?', ['%emails%'])->first();
        $receptionType = OccurrenceType::whereRaw('lower(name) like ?', ['%recep%'])->orWhereRaw('lower(name) like ?', ['%reception%'])->first();
        $callTypeId = $callType->id ?? null;
        $emailTypeId = $emailType->id ?? null;
    $receptionTypeId = $receptionType->id ?? null;

        // Aggregate occurrences by day for Calls and Emails. Use occurrence_type id when available,
        // otherwise fallback to matching occurrence type name substrings (call/chamad, email).
        // Calls
        $callRows = collect();
        if ($callTypeId) {
            $callRows = Occurrence::where('occurrence_type_id', $callTypeId)
                ->where('created_at', '>=', $dateStart)
                ->selectRaw("DATE(created_at) as day, count(*) as total")
                ->groupBy('day')
                ->get();
        }
        // Fallback by name if no rows found or type missing
        if ($callRows->isEmpty()) {
            $callRows = Occurrence::whereHas('type', function($q){
                $q->whereRaw('lower(name) like ?', ['%chamad%'])->orWhereRaw('lower(name) like ?', ['%call%']);
            })->where('created_at', '>=', $dateStart)
            ->selectRaw("DATE(created_at) as day, count(*) as total")
            ->groupBy('day')
            ->get();
        }

        foreach ($callRows as $r) {
            $d = \Carbon\Carbon::parse($r->day)->format('d/m');
            $index = array_search($d, $labels);
            if ($index !== false) $chartCalls['data'][$index] = (int) $r->total;
        }

        // Emails
        $emailRows = collect();
        if ($emailTypeId) {
            $emailRows = Occurrence::where('occurrence_type_id', $emailTypeId)
                ->where('created_at', '>=', $dateStart)
                ->selectRaw("DATE(created_at) as day, count(*) as total")
                ->groupBy('day')
                ->get();
        }
        if ($emailRows->isEmpty()) {
            $emailRows = Occurrence::whereHas('type', function($q){
                $q->whereRaw('lower(name) like ?', ['%email%'])->orWhereRaw('lower(name) like ?', ['%emails%']);
            })->where('created_at', '>=', $dateStart)
            ->selectRaw("DATE(created_at) as day, count(*) as total")
            ->groupBy('day')
            ->get();
        }

        foreach ($emailRows as $r) {
            $d = \Carbon\Carbon::parse($r->day)->format('d/m');
            $index = array_search($d, $labels);
            if ($index !== false) $chartEmails['data'][$index] = (int) $r->total;
        }

        // Prepare chart for attachments (docs) across all occurrences
        $chartDocs = ['labels' => $labels, 'data' => array_fill(0, $days, 0)];
        $rows = Attachment::where('created_at', '>=', $dateStart)
            ->selectRaw("DATE(created_at) as day, count(*) as total")
            ->groupBy('day')
            ->get();
        foreach ($rows as $r) {
            $d = \Carbon\Carbon::parse($r->day)->format('d/m');
            $index = array_search($d, $labels);
            if ($index !== false) $chartDocs['data'][$index] = (int) $r->total;
        }

        // Counts - occurrences grouped by type (admin scope)
        $counts = Occurrence::selectRaw('occurrence_type_id, count(*) as total')->groupBy('occurrence_type_id')->pluck('total','occurrence_type_id')->toArray();

        $attachmentsCount = Attachment::count();

        // Compute simple deltas (yesterday vs day before) for the admin (calls/emails/reception/attachments)
        $yesterday = now()->subDays(1)->startOfDay();
        $dayBefore = now()->subDays(2)->startOfDay();
        $deltaForType = function($typeId) use ($yesterday, $dayBefore) {
            if (! $typeId) return ['value' => 0, 'percent' => 0, 'trend' => 'same'];

            $last = Occurrence::where('occurrence_type_id', $typeId)
                ->where('created_at', '>=', $yesterday)
                ->where('created_at', '<', $yesterday->copy()->addDay())
                ->count();

            $prev = Occurrence::where('occurrence_type_id', $typeId)
                ->where('created_at', '>=', $dayBefore)
                ->where('created_at', '<', $dayBefore->copy()->addDay())
                ->count();

            $percent = 0; $trend = 'same';
            if ($prev == 0 && $last > 0) { $percent = 100; $trend = 'up'; }
            elseif ($prev == 0 && $last == 0) { $percent = 0; $trend = 'same'; }
            else { $diff = $last - $prev; $percent = $prev ? round(($diff / max(1,$prev)) * 100) : 0; $trend = $diff > 0 ? 'up' : ($diff < 0 ? 'down' : 'same'); }
            return ['value' => $last, 'percent' => $percent, 'trend' => $trend];
        };

        $deltaCalls = $deltaForType($callTypeId);
        $deltaEmails = $deltaForType($emailTypeId);
        $deltaReception = $deltaForType($receptionTypeId);

        // attachments delta
        $lastAtt = Attachment::where('created_at', '>=', $yesterday)->where('created_at', '<', $yesterday->copy()->addDay())->count();
        $prevAtt = Attachment::where('created_at', '>=', $dayBefore)->where('created_at', '<', $dayBefore->copy()->addDay())->count();
        $attDiff = $lastAtt - $prevAtt;
        if ($prevAtt == 0 && $lastAtt > 0) { $attPercent = 100; $attTrend = 'up'; }
        elseif ($prevAtt == 0 && $lastAtt == 0) { $attPercent = 0; $attTrend = 'same'; }
        else { $attPercent = $prevAtt ? round(($attDiff / max(1,$prevAtt)) * 100) : 0; $attTrend = $attDiff > 0 ? 'up' : ($attDiff < 0 ? 'down' : 'same'); }

        $deltaAttachments = ['value' => $lastAtt, 'percent' => $attPercent, 'trend' => $attTrend];

        // Render the admin dashboard view (design will match employee dashboard but with admin cards)
        return view('admin.dashboard', compact(
            'companiesCount','activeCompanies','employeesCount','occurrencesCount',
            'chartCompanies','chartCalls','chartEmails',
            'deltaCalls','deltaEmails','deltaReception','deltaAttachments'
        ));
    }
}
